import java.io.Serializable;
import java.util.HashSet;
import java.util.LinkedList; // implementing a queue
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.FileInputStream;
import java.io.ObjectInputStream;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
/**
 *  WebOfBeliefs - a basic WebOfBeliefs with:
 *  1. unstructured sentences (only the status of being a negation is relevant for revision)
 *  2. containing as nodes a subset of the sentences of a language (believed or not believed)
 *  3. containing with each sentence its negation as well (in case needed for a revision)
 *  4. nodes are never cut, beliefs may be surrendered (disbelieved), but are still present
 *  5. in expansions nodes are not added but switched from being disbelieved to being believed
 *  6. a static structure of nodes and their justifications (operative or not) in each step (i.e.
 *  beliefs and their justifications can be added, but are not in the algorithms of revision).
 *
 *  Algorithms of belief revision (i.e. Contraction, Expansion, Revision) work globally (i.e.
 *  move through the whole WebOfBeliefs).
 *  They work deterministically and without foresight (to a better choice of surrendering 
 *  beliefs), so, although Contraction looks at entrenchment of beliefs, the changes may not
 *  be minimal.
 *  
 *
 * @author MB
 * @version MAY 2021
 */
public class WebOfBeliefs implements Serializable, Cloneable
{
    private HashSet<Sentence> beliefs;
    private final int identifier;
    private int size = 0;
    /**
     * Constructor for objects of class WebOfBeliefs
     * @param - an identifier (e.g. for the file name if saved in a file)
     */
    public WebOfBeliefs(int identifier)
    {
        beliefs = new HashSet<Sentence>();
        this.identifier = identifier;
    }
    
    /**
     * Beliefs are always added with their negation as well, as one may need the
     * negation for revision. Adding a belief does not mean endorsing a belief.
     * If the added sentence comes with justifications their premises have to be
     * added to the WebOfBeliefs as well. The Justifications need not be operative.
     */
    public void addBelief(Sentence s){
         beliefs.add(s);
         beliefs.add(s.negation());
         size += 2;
         for(Justification j: s.giveJustifications()){
             for(Sentence premise: j.givePremises()){
                 if(!beliefs.contains(premise)){
                     addBelief(premise); // this may pass on further additions    
                 }
             }
         }
    }

    @Override
    public Object clone() throws CloneNotSupportedException{
        WebOfBeliefs clone = new WebOfBeliefs(identifier);
        for(Sentence s: beliefs){
            clone.addBelief(s.copy());  // copies of the beliefs are used to not interfere with further processing   
        }
        return clone;
    }

    /**
     * Global Contraction Algorithm
     * Given a belief to be surrendered, the support of this belief has to be undermined 
     * (following the links to justifications of this belief). 
     * Additional to this upward direction of undermining, Contraction
     * has also to pass the new status of the surrendered beliefs to the beliefs its supports 
     * in downward direction as these may have to be surrendered as well.
     * @param - the belief to be surrendered
     * @return - all beliefs surrendered in the contraction
     */
    public LinkedList<Sentence> contraction(Sentence s){
        s.surrender();
        LinkedList<Sentence> queue = new LinkedList<Sentence>();
        LinkedList<Sentence> surrendered = new LinkedList<Sentence>();
        queue.addLast(s);
        surrendered.addLast(s);
        while(!queue.isEmpty()){
             Sentence focus = queue.remove(); // head of the queue
             //upwards
             for(Justification j: focus.giveJustifications()){ // we have to look at the justification(s) of the belief given up
                 if(j.isOperative()){ // only operative justifications have to be surrendered
                    Sentence leastEntrenched = j.giveLeastEntrenched(); // the premise to be surrendered is the least entrenched belief
                    leastEntrenched.surrender(); 
                    queue.addLast(leastEntrenched); // this has to be passed on for further contraction
                    surrendered.addLast(leastEntrenched);
                    System.out.println(leastEntrenched);
                 }
             }
             // downwards
             for(Sentence supported: focus.giveSupportedSentences()){ // we look at the supported sentences
                 if(!supported.isBelieved()){ // sentences not believed are irrelevant for contraction
                     break;
                 }
                 for(Justification j: supported.giveJustifications()){ // and look at their justifications, whether all may become inoperative
                     if(j.isOperative()){ // inoperative justifications need not be contracted
                         if(j.givePremises().contains(focus)){ // crucial are justifications with the sentence surrendered
                             // the sentence has already been surrendered, so we also surrender the justification
                             j.surrender();
                         }
                     }
                 } 
                 // Now we look, whether all justifications have been surrendered
                 boolean allJustificationsInoperative = true;
                 for(Justification j: supported.giveJustifications()){
                     if(j.isOperative()){
                         allJustificationsInoperative = false;
                         break; // one justification is enough
                     }
                 }
                 if(allJustificationsInoperative){
                     supported.surrender();
                     queue.addLast(supported); // contraction has to be passed on
                     surrendered.addLast(supported);
                     System.out.println(supported);
                 }
             }
        }
        return surrendered;
    }
    
    public WebOfBeliefs copy(){
        try{
            WebOfBeliefs copy = (WebOfBeliefs) this.clone();
            return copy;
        }
        catch(Exception e){
            System.err.println(e);
        }
        return null;
    }
    
    /**
     * Global Expansion Algorithm
     * Support is passed down the supporting links to further beliefs.
     * @param - the sentence now to be believed
     * @return - the new beliefs generated by the expansion
     * 
     */
    public LinkedList<Sentence> expansion(Sentence s){
        s.endorse();
        LinkedList<Sentence> queue = new LinkedList<Sentence>();
        LinkedList<Sentence> newBeliefs = new LinkedList<Sentence>();
        queue.addLast(s);
        newBeliefs.addLast(s);
        while(!queue.isEmpty()){
             Sentence focus = queue.remove();   // get head of the queue
             for(Sentence supported: focus.giveSupportedSentences()){ // we look at the supported sentences
                 if(supported.isBelieved()){ // beliefs already held need not to be considered
                     break;
                 }
                 for(Justification j: supported.giveJustifications()){ // and look at their justifications, whether one may become operative
                     boolean allPremisesTrue = true;
                     for(Sentence p: j.givePremises()){ // if one premise is not endorsed, this justification is not operative                     
                         if(!p.isBelieved()){
                             allPremisesTrue = false;
                             break;
                         }
                     }
                     if(allPremisesTrue){
                         j.endorse(); // the justification still is or may become operative
                         if(!supported.isBelieved()){ // this sentence has not been believed, but should now be believed: expansion
                             supported.endorse();
                             newBeliefs.addLast(supported);
                             System.out.println(supported);
                             queue.addLast(supported); // the expansion with this sentence has to be passed on
                         }
                     }
                     // we cannot break here as more than one justification can become operative
                 }    
             }
        }
        return newBeliefs;
    }
    
    /**
     * Revision requires finding the negation of a sentence (by linear search).
     */
    private Sentence findNegation(Sentence s){
        String negatedContent = s.negation().getContent();
        for(Sentence belief: beliefs){
             if(belief.getContent().equals(negatedContent)){
                  return belief;   
             }
        }
        return null;
    }
    
    /**
     * @return - all beliefs, endorsed or not
     */
    public HashSet<Sentence> giveBeliefs(){
        return beliefs;
    }

    /**
     * @return - returns only the beliefs endorsed; the content of an agent's 'belief box'
     */
    public HashSet<Sentence> giveHeldBeliefs(){
         HashSet<Sentence> held = new HashSet<Sentence>();
         for(Sentence s: beliefs){
             if(s.isBelieved()){
                 held.add(s);
             }
         }
         return held;
    }
    
    /**
     * @return - a String describing the held beliefs and their operative justifications
     */
    public String heldBeliefsAndArguments(){
        int arguments = 1;
         String out = "WebOfBeliefs " + identifier + "\nStructure and content:\n";
         for(Sentence s: beliefs){
             if(s.isBelieved()){
                 out += s + "\n\t held because:\n ";
                 for(Justification j: s.giveJustifications()){
                     if(j.isOperative()){
                         out += "Argument " + (arguments++) + "\n";
                         for(Sentence p: j.givePremises()){
                             out += p + "\n";
                         }
                     }
                 }
             }
         }
         return out;
    }
    
    public int numberOfBeliefs(){
        return size;
    }
    
    public void print(){
         System.out.println(this);   
    }
    
    public void printHeldBeliefsAndArguments(){
         System.out.println(heldBeliefsAndArguments());   
    }
    
    public boolean restoreFromFile(String fileName){
        try{
            FileInputStream fi = new FileInputStream(new File(fileName));
            ObjectInputStream oi = new ObjectInputStream(fi);
            WebOfBeliefs web = (WebOfBeliefs) oi.readObject();
            oi.close();
            fi.close();
            beliefs = web.giveBeliefs();
            return true;
        }
        catch (FileNotFoundException e) {
            System.err.println("File not found");
        } catch (IOException e) {
            System.err.println("Error initializing stream");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }
    
    /**
     * A Global Revision Algorithm
     * Follows the 'Levi Identity' (that revision is contraction followed by expansion with
     * the negation).
     * @param - the belief to be switched
     */
    public boolean revision(Sentence s){
        WebOfBeliefs copyToRestore = this.copy(); // if somethings goes wrong during execution
        // Contraction
        if (contraction(s).size() > 0){
            // Contraction has (supposedly) succeeded, we now find the negation of the sentence
            Sentence negation = findNegation(s);
            if(negation==null){
                 // if the negation cannot be found, revision cannot procede. We restate the old WebOfBeliefs.
                 System.err.println("Negation could not be found in the WebOfBeliefs!");
                 beliefs = copyToRestore.giveBeliefs();
                 System.err.println("Old state of WebOfBeliefs restored.");
                 return false;
            }
            else{ // Expansion
                expansion(negation);
                return true;
            }
        }
        return false;   
    }
    
    /**
     * @return - a String describing the whole WebOfBeliefs (i.e. not only endorsed beliefs
     * and justifications)
     */
    @Override
    public String toString(){
         int arguments = 1;
         String out = "WebOfBeliefs " + identifier + "\nStructure and content:\n";
         for(Sentence s: beliefs){
                 out += s + "\n\t might be held because:\n ";
                 for(Justification j: s.giveJustifications()){
                         out += "Argument " + (arguments++) + "\n";
                         for(Sentence p: j.givePremises()){
                             out += p + "\n";
                         }
                 }
         }
         return out;
    }
    
    public boolean writeToFile(){
        try{
            FileOutputStream f = new FileOutputStream(new File("WebOfBeliefs" + identifier + ".txt"));
            ObjectOutputStream o = new ObjectOutputStream(f);
            o.writeObject(this);
            o.close();
            f.close();
            return true;
        }
        catch (FileNotFoundException e) {
            System.err.println("File not found");
        } catch (IOException e) {
            System.err.println("Error initializing stream");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;   
    }
}
