import java.util.HashSet;
import java.io.Serializable;
/**
 * Sentence - nodes in a WebOfBelief.
 * Have an unstructured content. Can be believed or not.
 * Entrenchment is measured by the number of other sentences supported by this sentence.
 * Sentences have justifications (sets of sentences) recognized by the agent in the WebOfBeliefs.
 *
 * @author MB
 * @version MAY 2021
 */
public class Sentence implements Cloneable, Serializable
{
    private boolean believed = false;
    private final String content;
    private int entrenchment = 0;
    private HashSet<Justification> supportedBy;
    private HashSet<Sentence> supporting;
    /**
     * Constructor for objects of class Sentence
     * @param - the sentence's content as a String (proper content is not evaluated)
     */
    public Sentence(String content)
    {
        this.content = content;
        supportedBy = new HashSet<Justification>();
        supporting = new HashSet<Sentence>();
    }

    /**
     * Adds a (further) justification for this sentence. The justification can be recognized
     * without being operative.
     */
    public void addSupport(Justification j){
         supportedBy.add(j);   
    }
    
    public void addSupportedBelief(Sentence s){
        supporting.add(s);
        entrenchment++;
    }
    
    @Override
    public Object clone() throws CloneNotSupportedException{
        Sentence clone = new Sentence(content);
        if(believed){
            clone.endorse();
        }
        clone.setJustifications(supportedBy);
        clone.setSupported(supporting);
        clone.resetEntrenchment(entrenchment);
        return clone;
    }

    public Sentence copy(){
        try{
            Sentence copy = (Sentence) this.clone();
            return copy;
        }
        catch(Exception e){
            System.err.println(e);
        }
        return null;
    }
    
    public void endorse(){
        believed = true;
    }
    
    public int entrenchmentLevel(){
        return entrenchment;
    }
    
    public String getContent(){
         return content;   
    }
    
    public HashSet<Justification> giveJustifications(){
         return supportedBy;   
    }
    
    public HashSet<Sentence> giveSupportedSentences(){
         return supporting;   
    }
    
    public boolean isBelieved(){
        return believed;
    }
    
    /**
     * @return - the negation of the sentence; applying double negation elimination if necessary.
     */
    public Sentence negation(){
         String negatedContent = "";
         if(content.startsWith("It is not")){
             negatedContent = content.substring(24); // the part after the negation
         }
         else{
             negatedContent = "It is not the case that " + content;
         }
         return new Sentence(negatedContent);
    }
    
    public void resetEntrenchment(int level){
         entrenchment = level;   
    }
    
    public void setJustifications(HashSet<Justification> support){
         supportedBy = support;   
    }
    
    public void setSupported(HashSet<Sentence> supporting){
         this.supporting = supporting; 
    }
    
    public void surrender(){
        believed = false;
    }
    
    @Override
    public String toString(){
         return "[" + content + "]" + ((isBelieved())? "is believed" : "is not believed") + ".";
    }
}
